/*
 *
 *  Copyright (C) 2022 Andrew Gegg
 *
 * 	This file is part of the Gardeners Notebook application
 *
 *  The Gardeners Notebook application is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/gpl.html>.
 *
 *
 */

package uk.co.gardennotebook;

import javafx.beans.binding.StringBinding;
import javafx.beans.property.ObjectProperty;
import javafx.beans.property.Property;
import javafx.beans.property.ReadOnlyStringWrapper;
import javafx.beans.property.SimpleObjectProperty;
import javafx.collections.FXCollections;
import javafx.collections.ObservableList;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.message.EntryMessage;
import uk.co.gardennotebook.fxbean.CommentBean;
import uk.co.gardennotebook.fxbean.INotebookBean;
import uk.co.gardennotebook.fxbean.PurchaseBean;
import uk.co.gardennotebook.fxbean.RetailerBean;
import uk.co.gardennotebook.spi.GNDBException;
import uk.co.gardennotebook.spi.INotebookEntry;
import uk.co.gardennotebook.spi.NotebookEntryType;
import uk.co.gardennotebook.util.SimpleMoney;

import java.time.LocalDate;
import java.util.Objects;
import java.util.ResourceBundle;

/**
 *	A specific implementation of DiaryBean for Purchase diary entries.
 *
 * @author Andy Gegg
 *	@version	3.0.5
 *	@since	3.0.5
 */
final class DiaryBeanPurchase extends DiaryBean
{
    private static final Logger LOGGER = LogManager.getLogger();

    private final PurchaseBean purchase;

    private Integer itemKey = 0;

    final ObjectProperty<LocalDate> dateProperty;

    private final ObjectProperty<RetailerBean> mainTitleProperty;	//	the title - the Retailer
    private final StringBinding mainTitleText;
    private final ReadOnlyStringWrapper subTitleProperty;	//	fixed text 'Total'
    private final StringBinding subTitleText;
    private final ObjectProperty<SimpleMoney> varietyProperty;	// the Total Price
    private final StringBinding varietyText;
    private final Property<?> detailProperty = null;    //  null text
    private final StringBinding detailText;

    private final SimpleObjectProperty<ObservableList<CommentBean>> comments =
            new SimpleObjectProperty<>(this, "comments", FXCollections.emptyObservableList());

    private final ResourceBundle resources = ResourceBundle.getBundle("notebook");

    DiaryBeanPurchase(PurchaseBean purchaseBean)
    {
        EntryMessage log4jEntryMsg = LOGGER.traceEntry("DiaryBeanPurchase: constructor(): PurchaseBean:{}", purchaseBean);

        purchase = purchaseBean;
        itemKey = purchase.getKey();

        dateProperty = purchase.dateProperty();

        mainTitleProperty = purchase.retailerProperty();
        mainTitleText = new StringBinding()
        {
            {
                super.bind(mainTitleProperty);
            }
            @Override
            protected String computeValue()
            {
                return purchase.getRetailer().getName();
            }

            @Override
            public void dispose()
            {
                super.unbind(mainTitleProperty);
            }
        };

        subTitleProperty = new ReadOnlyStringWrapper(resources.getString("lbl.total"));
        subTitleText = new StringBinding()
        {
            @Override
            protected String computeValue()
            {
                return resources.getString("lbl.total");
            }
        };

        varietyProperty = purchase.totalPriceProperty();
        varietyText = new StringBinding()
        {
            {
                super.bind(varietyProperty);
            }
            @Override
            protected String computeValue()
            {
                return purchase.getTotalPrice().toString();
            }

            @Override
            public void dispose()
            {
                super.unbind(varietyProperty);
            }
        };

//        detailProperty = null;
        detailText = new StringBinding()
        {
            @Override
            protected String computeValue()
            {
                return "";
            }
        };

        LOGGER.traceExit(log4jEntryMsg);
    }

    @Override
    INotebookBean getItem()
    {
        return purchase;
    }

    @Override
    INotebookEntry getBaseItem()
    {
        return purchase.get().orElse(null);
    }

    @Override
    Integer getKey()
    {
        return itemKey;
    }

    @Override
    NotebookEntryType getItemType()
    {
        return NotebookEntryType.PURCHASE;
    }

    @Override
    boolean canDelete() throws GNDBException
    {
        return purchase.canDelete();
    }

    /**
     * The date of this diary entry
     *
     * @return	the (value of) the dateProperty
     */
    @Override
    LocalDate getDate()
    {
        return dateProperty.getValue();
    }

    @Override
    void setDate(LocalDate date)
    {
        dateProperty().setValue(date);
    }

    @Override
    ObjectProperty<LocalDate> dateProperty()
    {
        return dateProperty;
    }

    /**
     * The title for this Diary entry - here, the Retailer's name
     *
     * @return	the (value of) the mainTitleProperty
     */
    @Override
    StringBinding getMainTitleText()
    {
        return mainTitleText;
    }

    @Override
    ObjectProperty<RetailerBean> mainTitleProperty()
    {
        return mainTitleProperty;
    }

    /**
     * The first detail for this Diary entry - here, the fixed text 'Total'
     *
     * @return	the (value of) the subTitleProperty
     */
    @Override
    StringBinding getSubTitleText()
    {
        return subTitleText;
    }

    @Override
    ReadOnlyStringWrapper subTitleProperty()
    {
        return subTitleProperty;
    }

    /**
     * The second detail for this Diary entry - here, the total price
     *
     * @return	the (value of) the varietyProperty
     */
    @Override
    StringBinding getVarietyText()
    {
        return varietyText;
    }

    @Override
    Property<SimpleMoney> varietyProperty()
    {
        return varietyProperty;
    }

    /**
     * The third detail for this Diary entry - here, null
     *
     * @return	the (value of) the detailProperty
     */
    @Override
    StringBinding getDetailText()
    {
        return detailText;
    }

    @Override
    Property<?> detailProperty()
    {
        return detailProperty;
    }

    private ObservableList<CommentBean> getCommentList()
    {
        return purchase.getComments();
    }

    @Override
    ObservableList<CommentBean> getComments()
    {
        LOGGER.debug("getComments(): getValue(): {}", commentProperty().getValue());
        return commentProperty().getValue();
    }

    @Override
    SimpleObjectProperty<ObservableList<CommentBean>> commentProperty()
    {
        LOGGER.debug("commentProperty(): getCommentList(): {}", getCommentList());
        comments.set(getCommentList());	//	2.9.6
        return comments;
    }

    @Override
    /**
     * Implement the comparison for sorting.
     * Items are first sorted in date order.
     * if same date, then by item type
     *      -   weather then husbandry, then affliction,
     * 		-	then groundwork, then wildlife,
     * 		-	then purchases and items within purchases,
     * 		-	then sales and sale items within sales
     * 		-	and journal entries
     * 	Items of same type on same day are sorted by key to give a stable ordering
     * 	    (this will usually correspond to sorting by order of creation).
     *
     * @implNote Be very careful with the sort order for Purchases - the sorting for PurchaseItems relies on it.
     */
    public int compareTo(DiaryBean o)
    {
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("compareTo(): this: {}. other: {}", this.toString(), o.toString());

//        LOGGER.debug("this.dateProperty: {}", this.dateProperty.get());
//        LOGGER.debug("other.dateProperty: {}", o.dateProperty().get());

        if (!this.dateProperty.get().equals(o.dateProperty().get()))
        {//sort on date as primary key
            return this.dateProperty.get().compareTo(o.dateProperty().get());
        }

        int retVal = switch (o.getItemType())
                {
                    case PURCHASE -> this.itemKey.compareTo(o.getKey()); //  stable ordering for entries of same type
                    case WEATHER, HUSBANDRY, AFFLICTIONEVENT, GROUNDWORK, WILDLIFE -> 1;
                    case PURCHASEITEM -> (purchase.getKey().equals(((DiaryBeanPurchaseItem)o).getPurchaseKey())) ? -1 :
                                purchase.getKey().compareTo(((DiaryBeanPurchaseItem)o).getPurchaseKey());
                    default -> -1;
                };
        LOGGER.debug("retVal: {}", retVal);
        return retVal;
    }

    @Override
    public boolean equals(Object o)
    {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;
        if (!super.equals(o)) return false;
        DiaryBeanPurchase that = (DiaryBeanPurchase) o;
        return purchase.equals(that.purchase) && itemKey.equals(that.itemKey) && mainTitleProperty.equals(that.mainTitleProperty) && mainTitleText.equals(that.mainTitleText) && subTitleProperty.equals(that.subTitleProperty) && subTitleText.equals(that.subTitleText) && varietyProperty.equals(that.varietyProperty) && varietyText.equals(that.varietyText) && Objects.equals(detailProperty, that.detailProperty) && Objects.equals(detailText, that.detailText) && Objects.equals(comments, that.comments) && Objects.equals(resources, that.resources);
    }

    @Override
    public int hashCode()
    {
        return Objects.hash(super.hashCode(), purchase, itemKey, mainTitleProperty, mainTitleText, subTitleProperty, subTitleText, varietyProperty, varietyText, detailProperty, detailText, comments, resources);
    }

    @Override
    public  String toString()
    {
        return ("DiaryBeanPurchase of: hash: " + this.hashCode() + ", " + purchase.toString());
    }

}
