/*
 *
 *  Copyright (C) 2022 Andrew Gegg
 *
 * 	This file is part of the Gardeners Notebook application
 *
 *  The Gardeners Notebook application is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/gpl.html>.
 *
 *
 */

package uk.co.gardennotebook;

import javafx.beans.binding.StringBinding;
import javafx.beans.property.ObjectProperty;
import javafx.beans.property.Property;
import javafx.beans.property.SimpleObjectProperty;
import javafx.collections.FXCollections;
import javafx.collections.ObservableList;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.message.EntryMessage;
import uk.co.gardennotebook.fxbean.CommentBean;
import uk.co.gardennotebook.fxbean.HusbandryBean;
import uk.co.gardennotebook.fxbean.HusbandryClassBean;
import uk.co.gardennotebook.fxbean.INotebookBean;
import uk.co.gardennotebook.fxbean.LocationBean;
import uk.co.gardennotebook.fxbean.PlantSpeciesBean;
import uk.co.gardennotebook.fxbean.PlantVarietyBean;
import uk.co.gardennotebook.spi.GNDBException;
import uk.co.gardennotebook.spi.INotebookEntry;
import uk.co.gardennotebook.spi.NotebookEntryType;

import java.time.LocalDate;
import java.util.Objects;

/**
 *	A specific implementation of DiaryBean for Husbandry diary entries.
 *
 * @author Andy Gegg
 *	@version	3.0.5
 *	@since	3.0.5
 */
final class DiaryBeanHusbandry extends DiaryBean
{
    private static final Logger LOGGER = LogManager.getLogger();

    private final HusbandryBean husbandry;

    private Integer itemKey = 0;

    final ObjectProperty<LocalDate> dateProperty;

    private final ObjectProperty<HusbandryClassBean> mainTitleProperty;	//	the title - the HusbandryClass
    private final StringBinding mainTitleText;
    private final Property<PlantSpeciesBean> subTitleProperty;	//	the PlantSpecies
    private final StringBinding subTitleText;
    private final Property<PlantVarietyBean> varietyProperty;	// the PlantVariety
    private final StringBinding varietyText;
    private final Property<LocationBean> detailProperty;    //  the Location
    private final StringBinding detailText;

    private final SimpleObjectProperty<ObservableList<CommentBean>> comments =
            new SimpleObjectProperty<>(this, "comments", FXCollections.emptyObservableList());

    DiaryBeanHusbandry(HusbandryBean husbandryBean)
    {
        EntryMessage log4jEntryMsg = LOGGER.traceEntry("DiaryBeanHusbandry: constructor(): HusbandryBean:{}", husbandryBean);

        husbandry = husbandryBean;
        itemKey = husbandry.getKey();

        dateProperty = husbandry.dateProperty();

        mainTitleProperty = husbandry.husbandryClassProperty();
        mainTitleText = new StringBinding()
        {
            {
                super.bind(mainTitleProperty);
            }
            @Override
            protected String computeValue()
            {
                return husbandry.getHusbandryClass().getName();
            }

            @Override
            public void dispose()
            {
                super.unbind(mainTitleProperty);
            }
        };

        subTitleProperty = husbandry.plantSpeciesProperty();
        subTitleText = new StringBinding()
        {
            {
                super.bind(subTitleProperty);
            }
            @Override
            protected String computeValue()
            {
                return husbandry.getPlantSpecies().getCommonName();
            }

            @Override
            public void dispose()
            {
                super.unbind(subTitleProperty);
            }
        };

        varietyProperty = husbandry.plantVarietyProperty();
        varietyText = new StringBinding()
        {
            {
                super.bind(varietyProperty);
            }
            @Override
            protected String computeValue()
            {
                if (!husbandry.hasPlantVariety()) return "";
                return husbandry.getPlantVariety().getCommonName();
            }

            @Override
            public void dispose()
            {
                super.unbind(varietyProperty);
            }
        };

        detailProperty = husbandry.locationProperty();
        detailText = new StringBinding()
        {
            {
                super.bind(detailProperty);
            }
            @Override
            protected String computeValue()
            {
                if (!husbandry.hasLocation()) return "";
                return husbandry.getLocation().getName();
            }

            @Override
            public void dispose()
            {
                super.unbind(detailProperty);
            }
        };

        LOGGER.traceExit(log4jEntryMsg);
    }

    @Override
    INotebookBean getItem()
    {
        return husbandry;
    }

    @Override
    INotebookEntry getBaseItem()
    {
        return husbandry.get().orElse(null);
    }

    @Override
    Integer getKey()
    {
        return itemKey;
    }

    @Override
    NotebookEntryType getItemType()
    {
        return NotebookEntryType.HUSBANDRY;
    }

    @Override
    boolean canDelete() throws GNDBException
    {
        return husbandry.canDelete();
    }

    /**
     * The date of this diary entry
     *
     * @return	the (value of) the dateProperty
     */
    @Override
    LocalDate getDate()
    {
        return dateProperty.getValue();
    }

    @Override
    void setDate(LocalDate date)
    {
        dateProperty().setValue(date);
    }

    @Override
    ObjectProperty<LocalDate> dateProperty()
    {
        return dateProperty;
    }

    /**
     * The title for this Diary entry - here, the HusbandryClass name
     *
     * @return	the (value of) the mainTitleProperty
     */
    @Override
    StringBinding getMainTitleText()
    {
        return mainTitleText;
    }

    @Override
    ObjectProperty<HusbandryClassBean> mainTitleProperty()
    {
        return mainTitleProperty;
    }

    /**
     * The first detail for this Diary entry - here, the PlantSpecies
     *
     * @return	the (value of) the subTitleProperty
     */
    @Override
    StringBinding getSubTitleText()
    {
        return subTitleText;
    }

    @Override
    Property<PlantSpeciesBean> subTitleProperty()
    {
        return subTitleProperty;
    }

    /**
     * The second detail for this Diary entry - here, the PlantVariety
     *
     * @return	the (value of) the varietyProperty
     */
    @Override
    StringBinding getVarietyText()
    {
        return varietyText;
    }

    @Override
    Property<PlantVarietyBean> varietyProperty()
    {
        return varietyProperty;
    }

    /**
     * The third detail for this Diary entry - here, the Location
     *
     * @return	the (value of) the detailProperty
     */
    @Override
    StringBinding getDetailText()
    {
        return detailText;
    }

    @Override
    Property<LocationBean> detailProperty()
    {
        return detailProperty;
    }

    private ObservableList<CommentBean> getCommentList()
    {
        return husbandry.getComments();
    }

    @Override
    ObservableList<CommentBean> getComments()
    {
        LOGGER.debug("getComments(): getValue(): {}", commentProperty().getValue());
        return commentProperty().getValue();
    }

    @Override
    SimpleObjectProperty<ObservableList<CommentBean>> commentProperty()
    {
        LOGGER.debug("commentProperty(): getCommentList(): {}", getCommentList());
        comments.set(getCommentList());	//	2.9.6
        return comments;
    }

    @Override
    /**
     * Implement the comparison for sorting.
     * Items are first sorted in date order.
     * if same date, then by item type
     *      -   weather then husbandry, then affliction,
     * 		-	then groundwork, then wildlife,
     * 		-	then purchases and items within purchases,
     * 		-	then sales and sale items within sales
     * 		-	and journal entries
     * 	Items of same type on same day are sorted by key to give a stable ordering
     * 	    (this will usually correspond to sorting by order of creation).
     */
    public int compareTo(DiaryBean o)
    {
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("compareTo(): this: {}. other: {}", this.toString(), o.toString());

//        LOGGER.debug("this.dateProperty: {}", this.dateProperty.get());
//        LOGGER.debug("other.dateProperty: {}", o.dateProperty().get());

        if (!this.dateProperty.get().equals(o.dateProperty().get()))
        {//sort on date as primary key
            return this.dateProperty.get().compareTo(o.dateProperty().get());
        }

        return switch (o.getItemType())
                {
                    case GROUNDWORK -> this.itemKey.compareTo(o.getKey()); //  stable ordering for entries of same type
                    case WEATHER, HUSBANDRY, AFFLICTIONEVENT -> 1;
                    default -> -1;
                };
    }

    @Override
    public boolean equals(Object o)
    {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;
        if (!super.equals(o)) return false;
        DiaryBeanHusbandry that = (DiaryBeanHusbandry) o;
        return husbandry.equals(that.husbandry) && itemKey.equals(that.itemKey) && dateProperty.equals(that.dateProperty) && mainTitleProperty.equals(that.mainTitleProperty) && mainTitleText.equals(that.mainTitleText) && subTitleProperty.equals(that.subTitleProperty) && subTitleText.equals(that.subTitleText) && Objects.equals(varietyProperty, that.varietyProperty) && Objects.equals(varietyText, that.varietyText) && Objects.equals(detailProperty, that.detailProperty) && Objects.equals(detailText, that.detailText) && Objects.equals(comments, that.comments);
    }

    @Override
    public int hashCode()
    {
        return Objects.hash(super.hashCode(), husbandry, itemKey, dateProperty, mainTitleProperty, mainTitleText, subTitleProperty, subTitleText, varietyProperty, varietyText, detailProperty, detailText, comments);
    }

    @Override
    public  String toString()
    {
        return ("DiaryBeanHusbandry of: hash: " + this.hashCode() + ", " + husbandry.toString());
    }

}
