/*
 *
 *  Copyright (C) 2021, 2023 Andrew Gegg
 *
 * 	This file is part of the Gardeners Notebook application
 *
 *  The Gardeners Notebook application is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/gpl.html>.
 *
 *
 */

/*
	Change log
    3.0.0   First version.
    3.1.0   Fixing the bug in CommentCellImpl meant that the comments tried to bind to the non-existent CroppingPlanBean for
            a Year node.  So, added a pretend bean and made sure the cell is not editable.
 */

package uk.co.gardennotebook;

import javafx.beans.property.*;
import javafx.collections.FXCollections;
import javafx.collections.ObservableList;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.message.EntryMessage;
import uk.co.gardennotebook.fxbean.*;
import uk.co.gardennotebook.spi.GNDBException;
import uk.co.gardennotebook.spi.NotebookEntryType;

import java.time.Year;

/**
 *	An aggregate and facade on the CroppingPlanBean and a place holder for the Year Of Plan.
 *
 *	@author	Andy Gegg
 *	@version	3.1.0
 *	@since	3.0.0
 */
final class CroppingPlanTreeBean
{

    private static final Logger LOGGER = LogManager.getLogger();

    private CroppingPlanBean croppingPlanBean = null;
//    private Year yearMarker;

    private final NotebookEntryType nodeType;    // CP for CroppingPlan, ?? for year marker (this is a bodge!)

    private StringProperty yearProperty = new SimpleStringProperty(this, "yearOfPlan", null);
    private ObjectProperty<LocationBean> locationProperty = new SimpleObjectProperty(this, "location", null);
    private ObjectProperty<CropRotationGroupBean> cropRotationGroupProperty = new SimpleObjectProperty(this, "cropRotationGroup", null);

    /**
     * Add a new Year node for the tree.
     *
     * @param yearOfPlan  the year to add
     */
    CroppingPlanTreeBean(Year yearOfPlan)
    {
        croppingPlanBean = new CroppingPlanBean();
        yearProperty.setValue(yearOfPlan.toString());
        nodeType = NotebookEntryType.UNKNOWN;

    }

    /**
     * Add a CroppingPlan node to the tree.
     *
     * @param croppingPlan    the Review to add
     */
    CroppingPlanTreeBean(CroppingPlanBean croppingPlan)
    {
        croppingPlanBean = croppingPlan;
        locationProperty = croppingPlan.locationProperty();
        cropRotationGroupProperty = croppingPlan.cropRotationGroupProperty();
        nodeType = NotebookEntryType.CROPPINGPLAN;
    }

    NotebookEntryType getNodeType()
    {
        return nodeType;
    }

    CroppingPlanBean getItem()
    {
        return croppingPlanBean;
    }

    ObservableList<CommentBean> getComments()
    {
        EntryMessage log4jEntryMsg = LOGGER.traceEntry("getComments()");

        if (nodeType == NotebookEntryType.CROPPINGPLAN)
        {
            return croppingPlanBean.getComments();
        }
        else
        {
           return FXCollections.emptyObservableList();
        }

    }

    boolean canDelete()
    {
        try {
            if (nodeType == NotebookEntryType.CROPPINGPLAN)
            {
                return croppingPlanBean.canDelete();
            }
            else
            {
                return false;
            }
        } catch (GNDBException ex) {
            PanicHandler.panic(ex);
        }
        return false;
    }

    void delete()
    {
        if (nodeType == NotebookEntryType.CROPPINGPLAN)
        {
            try {
                    croppingPlanBean.delete();
            } catch (GNDBException ex) {
                PanicHandler.panic(ex);
            }
        }
    }

    /**
     *	Wraps the YearOfPlan value of the underlying item
     *
     *	@return	a writable property wrapping the yearOfPlan attribute
     */
    public final StringProperty yearOfPlanProperty()
    {
        return yearProperty;
    }

    /**
     *	Wraps the Location value of the underlying item
     *
     *	@return	a writable property wrapping the location attribute
     */
    public final ObjectProperty<LocationBean> locationProperty()
    {
        return locationProperty;
    }

    /**
     *	Wraps the CropRotationGroup value of the underlying item
     *
     *	@return	a writable property wrapping the crop rotation group attribute
     */
    public final ObjectProperty<CropRotationGroupBean> cropRotationGroupProperty()
    {
        return cropRotationGroupProperty;
    }


}
