/*
 * Copyright (C) 2018-2021 Andrew Gegg
 *
 *	This file is part of the Garden Notebook application
 *
 * The Garden Notebook application is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/gpl.html>.
 */

/*
	Change log
	2.6.1   Code tidy
    2.9.6	When a Diary entry is added/changed, make sure updated comments are shown
 */

package uk.co.gardennotebook;

import uk.co.gardennotebook.spi.GNDBException;

import uk.co.gardennotebook.fxbean.AfflictionClassBean;

import java.util.Optional;
import java.io.IOException;
import java.util.ResourceBundle;
import java.util.function.BiConsumer;
import java.util.function.Consumer;

import javafx.collections.FXCollections;
import javafx.collections.ObservableList;
import javafx.beans.property.SimpleObjectProperty;
import javafx.event.ActionEvent;
import javafx.stage.WindowEvent;
import javafx.fxml.FXML;
import javafx.fxml.FXMLLoader;
import javafx.scene.text.Text;
import javafx.scene.Node;
import javafx.scene.control.Alert;
import javafx.scene.control.ButtonType;
import javafx.scene.control.Button;
import javafx.scene.control.TableColumn;
import javafx.scene.control.TableView;
import javafx.scene.control.cell.TextFieldTableCell;
import javafx.scene.control.MenuItem;
import javafx.scene.layout.AnchorPane;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.message.EntryMessage;

/**
	*	Controller class for create/update of AfflictionClass catalogue
	*
	*	@author Andy Gegg
	*	@version	2.9.6
	*	@since	1.0
 */
final class AfflictionClassCat extends AnchorPane implements INotebookLoadable
{

	private static final Logger LOGGER = LogManager.getLogger();

	@FXML
	private ResourceBundle resources;

	@FXML
	private TableView<AfflictionClassBean> tblCatalogue;
	@FXML
	private TableColumn<AfflictionClassBean, String> colName;
	@FXML
	private TableColumn<AfflictionClassBean, String> colDescription;
	@FXML
	private TableColumn<AfflictionClassBean, AfflictionClassBean> colComment;
	@FXML
	private Button btnChange;
	@FXML
	private Button btnDelete;

	@FXML
	private MenuItem ctxmnuDelete;

	private Consumer<Node> loadSplit;
	private Consumer<Node> clearSplit;
	private BiConsumer<String, Node> loadTab;
	private Consumer<Node> clearTab;

	AfflictionClassCat()
	{
		FXMLLoader fxmlLoader = new FXMLLoader(getClass().getResource("/fxml/AfflictionClassCat.fxml"),
			ResourceBundle.getBundle("notebook") );
		fxmlLoader.setRoot(this);
		fxmlLoader.setController(this);
		try {
			fxmlLoader.load();	// NB initialize is called from in here
		} catch (IOException exception) {
			throw new RuntimeException(exception);
		}
	}// constructor

	/**
	* Initializes the controller class.
	*/
	@FXML
	private void initialize()
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("initialize()");
		ObservableList<AfflictionClassBean> items = FXCollections.<AfflictionClassBean>emptyObservableList();
		try
		{
			items = AfflictionClassBean.fetchAll();
		} catch (GNDBException ex) {
			PanicHandler.panic(ex);
		}
		tblCatalogue.setColumnResizePolicy(NotebookResizer.using(tblCatalogue));
		colName.setCellFactory(TextFieldTableCell.forTableColumn());
		colName.setCellValueFactory(cdf -> cdf.getValue().nameProperty());
		colDescription.setCellFactory(TextFieldTableCell.forTableColumn());
		colDescription.setCellValueFactory(cdf -> cdf.getValue().descriptionProperty());
		colComment.setCellValueFactory((e)-> new SimpleObjectProperty<>( e.getValue()));
//		colComment.setCellFactory(x -> new EditorCommentTableCell(resources));
		colComment.setCellFactory(x -> new EditorCommentTableCell<>(resources));	//	2.9.6
		tblCatalogue.getItems().setAll(items);

		// only allow change and delete if there is a row selected
		btnChange.disableProperty().bind(tblCatalogue.getSelectionModel().selectedItemProperty().isNull());
		btnDelete.setDisable(true);
		tblCatalogue.getSelectionModel().selectedItemProperty().addListener((obs, old, nval) ->{
					try
					{
						btnDelete.setDisable( (nval == null) || !(nval.canDelete()) );
					} catch (GNDBException ex) {
						PanicHandler.panic(ex);
					}
				});

		if (items.size() <50)
		{
			double nameWidth = 0;
			for (AfflictionClassBean b : items)
			{
				Text t = new Text(b.getName());
				double wid = t.getLayoutBounds().getWidth();
				if (wid > nameWidth) nameWidth = wid;
			}
			colName.setPrefWidth(nameWidth+10);
		}

		LOGGER.traceExit(log4jEntryMsg);
	}	//	initialize()

	@Override
	public void setLoadSplit(Consumer<Node> code)
	{
		loadSplit = code;
	}

	@Override
	public void setClearSplit(Consumer<Node> code)
	{
		clearSplit = code;
	}

	@Override
	public void setLoadTab(BiConsumer<String, Node> code)
	{
		loadTab = code;
	}

	@Override
	public void setClearTab(Consumer<Node> code)
	{
		clearTab = code;
	}

	@FXML
	private void ctxmnuOnShowing(WindowEvent event)
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("ctxmnuOnShowing()");
		AfflictionClassBean ixBean = tblCatalogue.getSelectionModel().selectedItemProperty().get();
		try
		{
			ctxmnuDelete.setDisable( (ixBean == null) || !(ixBean.canDelete()) );
		} catch (GNDBException ex) {
			PanicHandler.panic(ex);
		}
		LOGGER.traceExit(log4jEntryMsg);
	}	//	ctxmnuOnShowing()

	@FXML
	private void btnAddOnAction(ActionEvent event)
	{
		ctxmnuAddOnAction(event);
	}

	@FXML
	private void btnChangeOnAction(ActionEvent event)
	{
		ctxmnuChangeOnAction(event);
	}

	@FXML
	private void btnDeleteOnAction(ActionEvent event)
	{
		ctxmnuDeleteOnAction(event);
	}

	@FXML
	private void ctxmnuAddOnAction(ActionEvent event)
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("ctxmnuAddOnAction()");
		AfflictionClassEditor tabCon = new AfflictionClassEditor();
		loadTab.accept(resources.getString("tab.afflictionclass"), tabCon);
		tabCon.newBeanProperty().addListener((obs, oldVal, newVal) -> {
			tblCatalogue.getItems().add(newVal);
		});
		LOGGER.traceExit(log4jEntryMsg);
	}	//	ctxmnuAddOnAction()

	@FXML
	private void ctxmnuChangeOnAction(ActionEvent event)
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("ctxmnuChangeOnAction()");
		AfflictionClassBean ixBean = tblCatalogue.getSelectionModel().selectedItemProperty().get();
		if (ixBean == null)
		{
			LOGGER.debug("thisValueBean is null");
			LOGGER.traceExit(log4jEntryMsg);
			return;
		}

		AfflictionClassEditor tabCon = new AfflictionClassEditor(ixBean);
		loadTab.accept(resources.getString("tab.afflictionclass"), tabCon);
		tabCon.deletedBeanProperty().addListener((obs, oldVal, newVal) -> {
			tblCatalogue.getItems().remove(ixBean);
		});
		LOGGER.traceExit(log4jEntryMsg);
	}	//	ctxmnuChangeOnAction()

	@FXML
	private void ctxmnuDeleteOnAction(ActionEvent event)
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("ctxmnuDeleteOnAction()");
		AfflictionClassBean ixBean = tblCatalogue.getSelectionModel().selectedItemProperty().get();
		if (ixBean == null)
		{
			LOGGER.debug("thisValueBean is null");
			LOGGER.traceExit(log4jEntryMsg);
			return;
		}

		boolean canDelete = false;
		try
		{
			canDelete = ixBean.canDelete();
		} catch (GNDBException ex) {
			PanicHandler.panic(ex);
		}
		if (!canDelete)
		{
			Alert checkDelete = new Alert(Alert.AlertType.INFORMATION, resources.getString("alert.cannotdelete"), ButtonType.OK);
			Optional<ButtonType> result = checkDelete.showAndWait();
			LOGGER.traceExit(log4jEntryMsg);
			return;
		}

		Alert checkDelete = new Alert(Alert.AlertType.CONFIRMATION, resources.getString("alert.confirmdelete"), ButtonType.NO, ButtonType.YES);
		Optional<ButtonType> result = checkDelete.showAndWait();
		LOGGER.debug("after delete dialog: result:{}, result.get:{}",result, result.get());
		if (result.isPresent() && result.get() == ButtonType.YES)
		{
			LOGGER.debug("after delete confirmed");
			try
			{
				ixBean.delete();
			} catch (GNDBException ex) {
				PanicHandler.panic(ex);
			}
			tblCatalogue.getItems().remove(ixBean);
		}
		LOGGER.traceExit(log4jEntryMsg);
	}	//	ctxmnuDeleteOnAction()


}

