/*
 *
 *  Copyright (C) 2021, 2022 Andrew Gegg
 *
 * 	This file is part of the Gardeners Notebook application
 *
 *  The Gardeners Notebook application is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/gpl.html>.
 *
 *
 */

/*
	Change log
	3.0.0   First version.
    3.0.4	Comment handling.
 */

package uk.co.gardennotebook.fxbean;

import javafx.beans.property.*;
import javafx.beans.value.ChangeListener;
import javafx.beans.value.ObservableValue;
import javafx.collections.FXCollections;
import javafx.collections.ObservableList;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.message.EntryMessage;
import uk.co.gardennotebook.spi.*;
import uk.co.gardennotebook.util.StoryLineTree;

import java.beans.PropertyChangeListener;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.Year;
import java.util.List;
import java.util.Optional;

/**
 *	Free text diary entries not connected to any specific plant, pest, etc.
 *
 *	@author	Andy Gegg
 *	@version	3.0.4
 *	@since	1.0
 */
public final class ReviewBean implements INotebookBean
{
    private static final Logger LOGGER = LogManager.getLogger();

    private IReview baseItem = null;

    private Integer itemKey = 0;
    private boolean newItem = false;
    private boolean explicitSave = false;
    private final SimpleBooleanProperty saveRequiredProperty = new SimpleBooleanProperty(this, "saveRequired", explicitSave);
    private IReviewBuilder explicitBuilder = null;

    // handle changes to the base item itself
    private PropertyChangeListener baseItemDeleted;
    private PropertyChangeListener baseItemReplaced;

    private final SimpleObjectProperty<LocalDate> dateProperty = new SimpleObjectProperty<>(this, "date", LocalDate.now());
    private final ChangeListener<LocalDate> dateListener = this::onDateChange;

    private final SimpleObjectProperty<Year> yearInReviewProperty = new SimpleObjectProperty<>(this, "yearInReview", Year.now());
    private final ChangeListener<Year> yearInReviewListener = this::onYearInReviewChange;

    private final SimpleObjectProperty<LocalDate> coverFromProperty = new SimpleObjectProperty<>(this, "coverFrom", null);
    private final ChangeListener<LocalDate> coverFromListener = this::onCoverFromChange;

    private final SimpleObjectProperty<LocalDate> coverToProperty = new SimpleObjectProperty<>(this, "coverTo", null);
    private final ChangeListener<LocalDate> coverToListener = this::onCoverToChange;

    private final SimpleStringProperty titleProperty = new SimpleStringProperty(this, "title", "");
    private final ChangeListener<String> titleListener = this::onTitleChange;

    private final SimpleStringProperty descriptionProperty = new SimpleStringProperty(this, "description", "");
    private final ChangeListener<String> descriptionListener = this::onDescriptionChange;

    private final ReadOnlyStringWrapper summaryProperty = new ReadOnlyStringWrapper(this, "summary", "");

    private final ReadOnlyObjectWrapper<LocalDateTime> lastUpdatedProperty = new ReadOnlyObjectWrapper<>(this, "lastUpdated", LocalDateTime.now());
    private final ReadOnlyObjectWrapper<LocalDateTime> createdProperty = new ReadOnlyObjectWrapper<>(this, "created", LocalDateTime.now());
    private ReadOnlyBooleanWrapper canDeleteProperty = null;
    private final ReadOnlyBooleanWrapper hasAncestorProperty = new ReadOnlyBooleanWrapper(this, "hasAncestor", false);
    private final ReadOnlyBooleanWrapper hasDescendantProperty = new ReadOnlyBooleanWrapper(this, "hasDescendant", false);

    private ReadOnlyBooleanWrapper isNewProperty = new ReadOnlyBooleanWrapper(this, "isNew", newItem);

    private BeanCommentHandler<IReview> beanCommentHandler;
    private final ReadOnlyStringWrapper commentTextProperty = new ReadOnlyStringWrapper(this, "commentText", "");

    /**
     *	Construct an 'empty' Bean.  Set the various property values then call save() to create the new ReviewBean
     */
    public ReviewBean()
    {
        this(null);
    }
    /**
     *	Construct a Bean wrapping the given Review
     *	If the parameter is null a new 'empty' Bean will be constructed
     *
     *	@param	initialValue	the Review to wrap.  If null an 'empty' bean will be constructed
     *
     */
    public ReviewBean(final IReview initialValue)
    {
        ChangeListener<Boolean> saveRequiredListener = (obs, old, nval) -> {
            if (nval && !explicitSave)
            {
                explicitSave = true;
                ITrug server = TrugServer.getTrugServer().getTrug();
                explicitBuilder = server.getReviewBuilder(baseItem);
            }
            if (!nval && explicitSave && (baseItem != null))
            {
                explicitSave = false;
                explicitBuilder = null;
            }
        };

        saveRequiredProperty.addListener(saveRequiredListener);

        if(initialValue == null)
        {
            newItem = true;
            //	add the listeners BEFORE setting values, or default values never get sent to the builder!
            addListeners();
            setDefaults();
            saveRequiredProperty.set(true);
            return;
        }

        baseItem = initialValue;

        setValues();

        addListeners();
        declareBaseListeners();
        addBaseListeners();
    }

    /**
     *	Returns all Review items wrapped as ReviewBean.
     *
     *	@return	a collection of ReviewBean beans
     *
     *	@throws	GNDBException	if the underlying persisted storage engine (e.g. database server) throws an exception
     *				The original error can be retrieved by <code>getCause()</code>
     */
    public static ObservableList<ReviewBean> fetchAll() throws GNDBException
    {
        EntryMessage log4jEntryMsg = LOGGER.traceEntry("fetchAll()");
        ITrug server = TrugServer.getTrugServer().getTrug();
        IReviewLister gal = server.getReviewLister();
        List<ReviewBean> ll = gal.fetch().stream().map(ReviewBean::new).toList();
        LOGGER.traceExit();
        return FXCollections.observableArrayList(ll);
    }

    /**
     *	Returns the underlying Review, if present
     *
     *	@return	the underlying Review, if present
     */
    public Optional<IReview> get()
    {
        return getValue();
    }

    /**
     *	Returns the underlying Review if present
     *
     *	@return	the underlying Review, if present
     */
    public Optional<IReview> getValue()
    {
        return Optional.ofNullable(baseItem);
    }

    @Override
    public Integer getKey()
    {
        return itemKey;
    }

    @Override
    public NotebookEntryType getType()
    {
        return NotebookEntryType.REVIEW;
    }

    @Override
    public boolean sameAs(INotebookBean other)
    {
        if (other == null || ((ReviewBean)other).baseItem == null || baseItem == null)
        {
            return false;
        }
        if (other.getType() != NotebookEntryType.REVIEW)
        {
            return false;
        }
        return baseItem.sameAs(((ReviewBean)other).baseItem);
    }

    @Override
    public boolean isNew()
    {
        return isNewProperty().get();
    }

    @Override
    public ReadOnlyBooleanProperty isNewProperty()
    {
        if (isNewProperty == null)
        {
            isNewProperty = new ReadOnlyBooleanWrapper(this, "isNew", newItem);
        }
        return isNewProperty.getReadOnlyProperty();
    }

    @Override
    public boolean canDelete() throws GNDBException
    {
        return canDeleteProperty().get();
    }

    @Override
    public ReadOnlyBooleanProperty canDeleteProperty() throws GNDBException
    {
        if (canDeleteProperty == null)
        {
            ITrug server = TrugServer.getTrugServer().getTrug();
            boolean canDel = server.getReviewBuilder(baseItem).canDelete();
            canDeleteProperty = new ReadOnlyBooleanWrapper(this, "canDelete", canDel);
        }
        return canDeleteProperty.getReadOnlyProperty();
    }

    @Override
    public boolean hasAncestor() throws GNDBException
    {
        //	Review items do not participate in story lines
        return false;
    }

    @Override
    public ReadOnlyBooleanProperty hasAncestorProperty() throws GNDBException
    {
        //	Review items do not participate in story lines
        return hasAncestorProperty.getReadOnlyProperty();
    }

    @Override
    public StoryLineTree<? extends INotebookBean> getAncestors() throws GNDBException
    {
        //	Review items do not participate in story lines
        return StoryLineTree.emptyTree();
    }

    @Override
    public boolean hasDescendant() throws GNDBException
    {
        //	Review items do not participate in story lines
        return false;
    }

    @Override
    public ReadOnlyBooleanProperty hasDescendantProperty() throws GNDBException
    {
        //	Review items do not participate in story lines
        return hasDescendantProperty.getReadOnlyProperty();
    }

    @Override
    public StoryLineTree<? extends INotebookBean> getDescendants() throws GNDBException
    {
        //	Review items do not participate in story lines
        return StoryLineTree.emptyTree();
    }

    public LocalDate getDate()
    {
        return dateProperty.get();
    }
    public void setDate(final LocalDate newVal)
    {
        dateProperty.set(newVal);
    }
    /**
     *	Wraps the Date value of the Review
     *
     *	@return	a writable property wrapping the date attribute
     */
    public ObjectProperty<LocalDate> dateProperty()
    {
        return dateProperty;
    }

    private void onDateChange(ObservableValue<? extends LocalDate> obs, LocalDate old, LocalDate nval)
    {
        EntryMessage log4jEntryMsg = LOGGER.traceEntry("onDateChange(): old={}, new={}", old, nval);
        if (explicitSave)
        {
            LOGGER.debug("onDateChange(): explicitSave");
            explicitBuilder.date(nval);
        }
        else
        {
            LOGGER.debug("onDateChange(): NOT explicitSave");
            ITrug server = TrugServer.getTrugServer().getTrug();
            try
            {
                baseItem = server.getReviewBuilder(baseItem).date(nval).save();
            } catch (GNDBException ex) {
                throw new GNDBRuntimeException(ex);
            }
        }
        LOGGER.traceExit(log4jEntryMsg);
    }

    public Year getYearInReview()
    {
        return yearInReviewProperty.get();
    }
    public void setYearInReview(final Year newVal)
    {
        yearInReviewProperty.set(newVal);
    }
    /**
     *	Wraps the Date value of the Review
     *
     *	@return	a writable property wrapping the date attribute
     */
    public ObjectProperty<Year> yearInReviewProperty()
    {
        return yearInReviewProperty;
    }

    private void onYearInReviewChange(ObservableValue<? extends Year> obs, Year old, Year nval)
    {
        EntryMessage log4jEntryMsg = LOGGER.traceEntry("onYearInReviewChange(): old={}, new={}", old, nval);
        if (explicitSave)
        {
            LOGGER.debug("onYearInReviewChange(): explicitSave");
            explicitBuilder.yearInReview(nval);
        }
        else
        {
            LOGGER.debug("onYearInReviewChange(): NOT explicitSave");
            ITrug server = TrugServer.getTrugServer().getTrug();
            try
            {
                baseItem = server.getReviewBuilder(baseItem).yearInReview(nval).save();
            } catch (GNDBException ex) {
                throw new GNDBRuntimeException(ex);
            }
        }
        LOGGER.traceExit(log4jEntryMsg);
    }

    public LocalDate getCoverFrom()
    {
        return coverFromProperty.get();
    }
    public void setCoverFrom(final LocalDate newVal)
    {
        coverFromProperty.set(newVal);
    }
    /**
     *	Wraps the Date value of the Review
     *
     *	@return	a writable property wrapping the date attribute
     */
    public ObjectProperty<LocalDate> coverFromProperty()
    {
        return coverFromProperty;
    }

    private void onCoverFromChange(ObservableValue<? extends LocalDate> obs, LocalDate old, LocalDate nval)
    {
        EntryMessage log4jEntryMsg = LOGGER.traceEntry("onCoverFromChange(): old={}, new={}", old, nval);
        if (explicitSave)
        {
            LOGGER.debug("onCoverFromChange(): explicitSave");
            explicitBuilder.coverFrom(nval);
        }
        else
        {
            LOGGER.debug("onCoverFromChange(): NOT explicitSave");
            if (nval != null && coverToProperty.get() != null && nval.isAfter(getCoverFrom()))
            {
                coverFromProperty.set(old);
            }
            ITrug server = TrugServer.getTrugServer().getTrug();
            try
            {
                baseItem = server.getReviewBuilder(baseItem).coverFrom(nval).save();
            } catch (GNDBException ex) {
                throw new GNDBRuntimeException(ex);
            }
        }
        LOGGER.traceExit(log4jEntryMsg);
    }

    public LocalDate getCoverTo()
    {
        return coverToProperty.get();
    }
    public void setCoverTo(final LocalDate newVal)
    {
        coverToProperty.set(newVal);
    }
    /**
     *	Wraps the Date value of the Review
     *
     *	@return	a writable property wrapping the date attribute
     */
    public ObjectProperty<LocalDate> coverToProperty()
    {
        return coverToProperty;
    }

    private void onCoverToChange(ObservableValue<? extends LocalDate> obs, LocalDate old, LocalDate nval)
    {
        EntryMessage log4jEntryMsg = LOGGER.traceEntry("onCoverToChange(): old={}, new={}", old, nval);
        if (explicitSave)
        {
            LOGGER.debug("onCoverToChange(): explicitSave");
            explicitBuilder.coverTo(nval);
        }
        else
        {
            LOGGER.debug("onCoverToChange(): NOT explicitSave");
            if (nval != null && coverFromProperty.get() != null && nval.isBefore(getCoverFrom()))
            {
                coverToProperty.set(old);
            }
            ITrug server = TrugServer.getTrugServer().getTrug();
            try
            {
                baseItem = server.getReviewBuilder(baseItem).coverTo(nval).save();
            } catch (GNDBException ex) {
                throw new GNDBRuntimeException(ex);
            }
        }
        LOGGER.traceExit(log4jEntryMsg);
    }

    public String getTitle()
    {
        return titleProperty.get();
    }
    public void setTitle(final String newVal)
    {
        titleProperty.set(newVal);
    }
    /**
     *	Wraps the Title value of the Review
     *
     *	@return	a writable property wrapping the variety attribute
     */
    public StringProperty titleProperty()
    {
        return titleProperty;
    }

    private void onTitleChange(ObservableValue<? extends String> obs, String old, String nval)
    {
        EntryMessage log4jEntryMsg = LOGGER.traceEntry("onTitleChange(): old={}, new={}", old, nval);

        if (explicitSave)
        {
            LOGGER.debug("onTitleChange(): explicitSave");
            explicitBuilder.title(nval);
        }
        else
        {
            LOGGER.debug("onTitleChange(): NOT explicitSave");
            if (((nval == null) || nval.isBlank()) && ((old != null) && !old.isBlank()))
            {
                titleProperty.set(old);
                return;
            }
            ITrug server = TrugServer.getTrugServer().getTrug();
            try
            {
                baseItem = server.getReviewBuilder(baseItem).title(nval).save();
            } catch (GNDBException ex) {
                throw new GNDBRuntimeException(ex);
            }
        }
        LOGGER.traceExit(log4jEntryMsg);
    }

    public String getDescription()
    {
        return descriptionProperty.get();
    }
    public void setDescription(final String newVal)
    {
        descriptionProperty.set(newVal);
    }
    /**
     *	Wraps the Description value of the Review
     *
     *	@return	a writable property wrapping the variety attribute
     */
    public StringProperty descriptionProperty()
    {
        return descriptionProperty;
    }

    private void onDescriptionChange(ObservableValue<? extends String> obs, String old, String nval)
    {
        EntryMessage log4jEntryMsg = LOGGER.traceEntry("onDescriptionChange(): old={}, new={}", old, nval);
        if (explicitSave)
        {
            LOGGER.debug("onDescriptionChange(): explicitSave");
            explicitBuilder.description(nval);
        }
        else
        {
            LOGGER.debug("onDescriptionChange(): NOT explicitSave");
            ITrug server = TrugServer.getTrugServer().getTrug();
            try
            {
                baseItem = server.getReviewBuilder(baseItem).description(nval).save();
            } catch (GNDBException ex) {
                throw new GNDBRuntimeException(ex);
            }
        }
        summaryProperty.set(deriveSummary(getDescription()));
//        System.out.println("onDescriptionChange: summary: "+getSummary());
        LOGGER.traceExit(log4jEntryMsg);
    }

    public String getSummary()
    {
        return summaryProperty.getReadOnlyProperty().get();
    }

    /**
     *	Wraps the Summary value of the Review
     *
     *	@return	a read-only property wrapping the Summary attribute
     */
    public ReadOnlyStringProperty summaryProperty()
    {
        return summaryProperty.getReadOnlyProperty();
    }

    private String deriveSummary(String htmlText)
    {
        EntryMessage log4jEntryMsg = LOGGER.traceEntry("deriveSummary(): htmlText: {}", htmlText);
        String workArea = htmlText;
        workArea = workArea.replaceAll("((<.*?>)+?)|((&.*?;)+?)|([\\r\\n]+?)", " ").strip();
        if (workArea.length() > 256)
            workArea = workArea.substring(0,256) + "...";
        LOGGER.debug("workArea: after replace: {}", workArea);

        return workArea;
    }

    public LocalDateTime getLastUpdated()
    {
        return lastUpdatedProperty.get();
    }
    /**
     *	Wraps the LastUpdated value of the Review entry.
     *	Note that this value cannot be changed by the user.
     *
     *	@return	a read-only property wrapping the lastUpdated attribute
     */
    public ReadOnlyObjectProperty<LocalDateTime> lastUpdatedProperty()
    {
        return lastUpdatedProperty.getReadOnlyProperty();
    }

    public LocalDateTime getCreated()
    {
        return createdProperty.get();
    }
    /**
     *	Wraps the Created value of the Review entry.
     *	Note that this value cannot be changed by the user.
     *
     *	@return	a read-only property wrapping the created attribute
     */
    public ReadOnlyObjectProperty<LocalDateTime> createdProperty()
    {
        return createdProperty.getReadOnlyProperty();
    }

    @Override
    public ObservableList<CommentBean> getComments()
    {
        EntryMessage log4jEntryMsg = LOGGER.traceEntry("getComments()");

        return LOGGER.traceExit(log4jEntryMsg, beanCommentHandler.getComments());
    }

    @Override
    public ReadOnlyStringProperty commentTextProperty()
    {
//        return beanCommentHandler.commentTextProperty();
        commentTextProperty.set(beanCommentHandler.commentTextProperty().get());
        return commentTextProperty;
    }

    @Override
    public void addComment(final String text) throws GNDBException
    {
        EntryMessage log4jEntryMsg = LOGGER.traceEntry("addComment({})", text);
        if (text == null || text.isBlank()) return;	//	2.9.6

        beanCommentHandler.addComment(text);	//	2.9.6

        if (explicitSave)
        {
            LOGGER.debug("addComment(): explicitSave");
        }
        else
        {	//	this cannot be a new instance of the parent Review
            LOGGER.debug("addComment(): NOT explicitSave");
            ITrug server = TrugServer.getTrugServer().getTrug();
            baseItem = server.getReviewBuilder(baseItem).addComment(text).save();
            setValues();	//	2.9.6
        }
        LOGGER.traceExit(log4jEntryMsg);
   }

    @Override
    public void addComment(CommentBean comment) throws GNDBException
    {
        EntryMessage log4jEntryMsg = LOGGER.traceEntry("addComment(comment bean): comment: {}, text : {}", comment, comment==null ? "null" :comment.getComment());
        if (comment == null) return;
        if (comment.getParentType() != this.getType()) return;
        if (comment.getComment() == null || comment.getComment().isBlank()) return;

        beanCommentHandler.addComment(comment);

        if (explicitSave)
        {
            LOGGER.debug("addComment(): explicitSave");
        }
        else
        {	//	this cannot be a new instance of the parent Review
            LOGGER.debug("addComment(): NOT explicitSave");
            ITrug server = TrugServer.getTrugServer().getTrug();
            baseItem = server.getReviewBuilder(baseItem).addComment(comment.getComment()).save();
            setValues();	//	2.9.6
        }
        LOGGER.debug("addComment(comment bean): commentTextProperty: {}", commentTextProperty().get());

        LOGGER.traceExit(log4jEntryMsg);
    }

    @Override
    public void changeCommentText(final CommentBean comment, final String text) throws GNDBException
    {
        EntryMessage log4jEntryMsg = LOGGER.traceEntry("changeCommentText(): comment={}, text={}", comment, text);
        if (text == null || text.isBlank()) return;

        if (comment == null)	//	2.9.6
        {
            addComment(text);
            return;
        }

        if (comment.getParentType() != this.getType()) return;

        beanCommentHandler.changeCommentText(comment, text);

        if (explicitSave)
        {
            LOGGER.debug("changeCommentText(): explicitSave");
        }
        else
        {	//	this cannot be a new instance of the parent Review
            LOGGER.debug("changeCommentText(): NOT explicitSave");
            ITrug server = TrugServer.getTrugServer().getTrug();
            baseItem = server.getReviewBuilder(baseItem).changeComment(comment.get().get(), text).save();
            setValues();	//	2.9.6
        }
        LOGGER.traceExit(log4jEntryMsg);
    }

    @Override
    public void changeCommentDate(CommentBean comment, final LocalDate date) throws GNDBException
    {
        EntryMessage log4jEntryMsg = LOGGER.traceEntry("changeCommentDate(): comment={}, date={}", comment, date);
        if (date == null) return;

        if (comment == null)
        {
            return;
        }

        if (comment.getParentType() != this.getType()) return;

        beanCommentHandler.changeCommentDate(comment, date);

        if (explicitSave)
        {
            LOGGER.debug("changeCommentDate(): explicitSave");
        }
        else
        {	//	this cannot be a new instance of the parent Review
            LOGGER.debug("changeCommentDate(): NOT explicitSave");
            ITrug server = TrugServer.getTrugServer().getTrug();
            baseItem = server.getReviewBuilder(baseItem).changeComment(comment.get().get(), date).save();
            setValues();	//	2.9.6
        }
        LOGGER.traceExit(log4jEntryMsg);
    }

    @Override
    public void deleteComment(CommentBean comment) throws GNDBException
    {
        EntryMessage log4jEntryMsg = LOGGER.traceEntry("deleteComment(): comment={}", comment);
        if (comment == null) return;

        if (comment.getParentType() != this.getType()) return;

        beanCommentHandler.deleteComment(comment);

        if (explicitSave)
        {
            LOGGER.debug("deleteComment(): explicitSave");
        }
        else
        {	//	this cannot be a new instance of the parent Review
            LOGGER.debug("deleteComment(): NOT explicitSave");
            ITrug server = TrugServer.getTrugServer().getTrug();
            baseItem = server.getReviewBuilder(baseItem).deleteComment(comment.get().get()).save();
            setValues();	//	2.9.6
        }
        LOGGER.traceExit(log4jEntryMsg);
    }

    public boolean isSaveRequired()
    {
        return explicitSave;
    }
    public void setSaveRequired(boolean reqd)
    {
        saveRequiredProperty.set(reqd);
    }
    public BooleanProperty saveRequiredProperty()
    {
        return saveRequiredProperty;
    }

    public boolean needSave()
    {
        if (!explicitSave)
            return false;

        return explicitBuilder.needSave() || beanCommentHandler.needSave();
    }

    public boolean canSave()
    {
        if (!explicitSave)
            return ((titleProperty.get() != null) && !titleProperty.get().isBlank());

        return explicitBuilder.canSave();
    }

    /**
     *	Save changes to the underlying Review item
     *
     *	@throws	GNDBException	if the underlying persisted storage engine (e.g. database server) throws an exception
     *				The original error can be retrieved by <code>getCause()</code>
     */
    public void save() throws GNDBException
    {
        EntryMessage log4jEntryMsg = LOGGER.traceEntry("save()");
        if (!explicitSave) return;

        saveComments();	//	do this here so that explicitBuilder knows there's a change

        if (!explicitBuilder.needSave())
        {
            return;
        }
        if (!explicitBuilder.canSave())
        {
            throw new IllegalStateException("ReviewBean: cannot save at this time - mandatory values not set");
        }

        baseItem = explicitBuilder.save();
        LOGGER.debug("save(): after explicitBuilder.save(): comments: {}", ()->baseItem.getComments());	//	2.9.6
        setValues();	//	2.9.6
//        LOGGER.debug("save(): after setValues(): comments: {}", ()->beanCommentHandler.getComments());
//        LOGGER.debug("save(): after setValues(): commentTextProperty: {}", ()->beanCommentHandler.commentTextProperty());
        saveRequiredProperty.set(false);
        LOGGER.traceExit(log4jEntryMsg);
    }	//	save()

    private void saveComments()
    {
        EntryMessage log4jEntryMsg = LOGGER.traceEntry("saveComments()");

        beanCommentHandler.saveComments(
                cb -> explicitBuilder.addComment(cb.getComment()),	//	add
                cb -> explicitBuilder.changeComment(cb.get().get(), cb.getComment()),	//	change text
                cb -> explicitBuilder.changeComment(cb.get().get(), cb.getDate()),	//	change date
                cb -> explicitBuilder.deleteComment(cb.get().get())		//	delete
        );
    }

    /**
     *	Delete the underlying Review entry.
     *
     *	@throws	GNDBException	if the underlying persisted storage engine (e.g. database server) throws an exception
     *				The original error can be retrieved by <code>getCause()</code>
     */
    public void delete() throws GNDBException
    {
        EntryMessage log4jEntryMsg = LOGGER.traceEntry("delete()");
        if (newItem) return;

        if (explicitSave)
        {
            explicitBuilder.delete();
            saveRequiredProperty.set(false);
        }
        else
        {
            ITrug server = TrugServer.getTrugServer().getTrug();
            server.getReviewBuilder(baseItem).delete();
        }
    }	//	delete()

    public void cancelEdit()
    {
        if (!explicitSave) return;
        if (newItem) return;

        saveRequiredProperty.set(false);
        setValues();
    }

    private void setDefaults()
    {
        saveRequiredProperty.setValue(false);
        dateProperty.setValue(LocalDate.now());
        yearInReviewProperty.setValue(Year.now());
        coverFromProperty.setValue(null);
        coverToProperty.setValue(null);
        titleProperty.setValue("");
        descriptionProperty.setValue("");
        lastUpdatedProperty.setValue(LocalDateTime.now());
        createdProperty.setValue(LocalDateTime.now());

        isNewProperty.set(true);
        beanCommentHandler = new BeanCommentHandler<>(this, baseItem);
        commentTextProperty.set(beanCommentHandler.commentTextProperty().get());
    }

    private void setValues()
    {
        saveRequiredProperty.setValue(false);

        dateProperty.setValue(baseItem.getDate());
        yearInReviewProperty.setValue(baseItem.getYearInReview());
        coverFromProperty.setValue(baseItem.getCoverFrom().orElse(null));
        coverToProperty.setValue(baseItem.getCoverTo().orElse(null));
        titleProperty.setValue(baseItem.getTitle());
        descriptionProperty.setValue(baseItem.getDescription().orElse(""));
        summaryProperty.set(deriveSummary(descriptionProperty.get()));
        lastUpdatedProperty.setValue(baseItem.getLastUpdated());
        createdProperty.setValue(baseItem.getCreated());

        itemKey = baseItem.getKey();
        newItem = false;
        isNewProperty.set(false);	//	2.9.6

        LOGGER.debug("setvalues(): about to change BeanCommentHandler");
        beanCommentHandler = new BeanCommentHandler<>(this, baseItem);
        commentTextProperty.set(beanCommentHandler.commentTextProperty().get());
    }

    private void addListeners()
    {
        dateProperty.addListener(dateListener);
        yearInReviewProperty.addListener(yearInReviewListener);
        coverFromProperty.addListener(coverFromListener);
        coverToProperty.addListener(coverToListener);
        titleProperty.addListener(titleListener);
        descriptionProperty.addListener(descriptionListener);
    }
    private void removeListeners()
    {
        dateProperty.removeListener(dateListener);
        yearInReviewProperty.removeListener(yearInReviewListener);
        coverFromProperty.removeListener(coverFromListener);
        coverToProperty.removeListener(coverToListener);
        titleProperty.removeListener(titleListener);
        descriptionProperty.removeListener(descriptionListener);
    }

    private void declareBaseListeners()
    {
        // handle changes to the base item itself
        baseItemDeleted = evt -> {
            removeListeners();
            removeBaseListeners();
            baseItem = null;
            setDefaults();
        };
        baseItemReplaced = evt -> {
            if (evt.getNewValue() != null)
            {
                removeBaseListeners();
                baseItem = (IReview) (evt.getNewValue());
                setValues();
                addBaseListeners();
            }
        };

    }

    private void addBaseListeners()
    {
        if (baseItem == null) return;
        baseItem.addPropertyChangeListener("deleted", baseItemDeleted);
        baseItem.addPropertyChangeListener("replaced", baseItemReplaced);

    }
    private void removeBaseListeners()
    {
        if (baseItem == null) return;
        baseItem.removePropertyChangeListener("deleted", baseItemDeleted);
        baseItem.removePropertyChangeListener("replaced", baseItemReplaced);

    }

    @Override
    public String toString()
    {
        return "ReviewBean wrapping " + baseItem;
    }

}
